#' Beta distributions : method of moments estimation 
#'
#' @description 
#' Beta(a,b) on (0,1) or (-1,1): map from moments (mean, SD) to (a,b)
#'
#' @param mu mean parameter, could be sample mean of data 
#' @param s SD parameter, could be sample SD of data 
#' @param imp1 indicator of minus/plus1, TRUE if interval is (-1,1), FALSE if (0,1).
#' Note that s^2<1-mu^2 for (-1,1) and s^2<mu*(1-mu) for (0,1)
#'
#' @return 
#' list with (a,b,a+b) where a,b come from method of moments estimation
#'
#' @examples
#' beta_mom2ab(0.4,0.2,imp1=TRUE)
#' beta_mom2ab(0.4,0.2,imp1=FALSE)
#'
#' @export
#'
beta_mom2ab = function(mu,s,imp1=FALSE)
{ v = s^2
  if(imp1) # ab>0 implies ab>0, v<4*mu0*(1-mu0)= = 1-mu^2
  { if(any(v>=1-mu^2)) { message("need s^2<1-mu^2"); return(NA) }
    mu0 = (mu+1)/2 # convert to mean of (x+1)/2
    ab = 4*mu0*(1-mu0)/v-1 
    a = mu0*ab 
  }
  else # ab>0 implies v<mu*(1-mu)
  { if(any(v>=mu*(1-mu))) { message("need s^2<mu*(1-mu)"); return(NA) }
    ab = mu*(1-mu)/v-1; a = ab*mu 
  }
  b = ab-a
  #if(iprint) print(cbind(mu,s,a,b,ab))
  cbind(a,b,ab)
}

#' QQ plot for Beta(a,b) distribution on (0,1) or (-1,1)
#'
#' @description 
#' QQ plot for Beta(a,b) distribution on (0,1) or (-1,1)
#'
#' @param x data vector on (0,1) or (-1,1)
#' @param a alpha parameter>0
#' @param b beta parameter>0
#' @param pos1 first subscript of correlation
#' @param pos2 second subscript of correlation
#'
#' @return
#' a QQ plot is produced, 0 is returned
#'
#' @examples
#' set.seed(1234)
#' n = 1000
#' x = rbeta(n,3,2)
#' mn = mean(x); s = sd(x)
#' ab = beta_mom2ab(mn,s)
#' qqbeta(x,ab[1],ab[2])
#' set.seed(12343)
#' n = 1000
#' x = 2*rbeta(n,5,3)-1
#' mn = mean(x); s = sd(x)
#' ab = beta_mom2ab(mn,s,imp1=TRUE)
#' qqbeta(x,ab[1],ab[2])
#'
#' @export
#'
qqbeta = function(x,a,b,pos1=0,pos2=0)
{ xs = sort(x)
  n = length(x)
  imp1 = FALSE
  if(xs[1]<= -1) { message("min value < -1"); return(0) }
  if(xs[n]>= 1) { message("max value > 1"); return(0) }
  if(xs[1]<0) imp1 = TRUE
  pp = ((1:n)-0.5)/n
  qq = qbeta(pp,a,b)
  if(imp1) qq = 2*qq-1
  plot(qq,xs,ylab="sample quantiles",xlab="beta quantiles")
  if(pos1>0) 
  { title(paste("QQ plot a,b=",round(a,1),round(b,1), " pos",pos1,pos2)) }
  else
  { title(paste("QQ plot a,b=",round(a,1),round(b,1))) }
  invisible(0)
}


