# C-vine with distribution of partial correlations conditional on previous trees

#' Generate Beta(afix,b) random variate on interval (qq,1) with mean mu
#'
#' @description
#' Generate Beta(afix,b) random variate on interval (qq,1) with mean mu
#'
#' @param qq value in interval (-1,1)
#' @param mu desired mean of Beta random variable in interval (qq,1)
#' @param afix positive value, first parameter of Beta random variable
#'
#' @return Beta(afix,b) random variate on interval (qq,1) with mean mu
#'
rBeta_shift = function(qq,mu,afix)
{ if(qq>=mu) r = runif(1,qq,1)
  else
  { mustar = (mu-qq)/(1-qq); b = afix*(1-mustar)/mustar
    w = rbeta(1,afix,b)
    r = qq+(1-qq)*w
  }
  return(r)
}

#' Generate random correlation matrix with all positive values
#' based on C-vine with distribution of partial correlations conditional on previous trees
#'
#' @description
#' Generate random correlation matrix with all positive values
#' based on C-vine with distribution of partial correlations conditional on previous trees
#' permutation not done here, r[d-1,d] is used to check validity
#'
#' @param d dimension of correlation matrix, >=4
#' @param a1 first parameter of Beta(a1,b1) random variable in (0,1) for row 1
#' @param b1 second parameter of Beta(a1,b1) random variable in (0,1) for row 1
#' @param muvec vector of desired means of partial correlations for trees 1,2,...,d-1 of C-vine
#' @param afix fixed first parameter for Beta random variables in trees 2 to d-1 
#' @param ipermute permute flag, if TRUE row/column permutation is applied
#' @param iprint flag for printing intermediate results if TRUE
#'
#' @return list with rr=correlation matrix ,pc=partial correlation matrix, 
#'   qm=matrix of q values (lower support points for partial correlations)
#'
#' @examples
#' To add
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines
#' Journal of Multivariate Analysis
#' 
#' @export
#'
rposcorr_meth2 = function(d,a1,b1,muvec,afix,ipermute=TRUE,iprint=FALSE)
{ #d>=4
  d = floor(d)
  if(d<4) { message("d should be >=4"); return(0) }
  r = matrix(1,d,d) # for random correlation matrix
  pc = matrix(0,d,d) # for partial correlations 
  # space for variables used in recursion algorithm
  ss = matrix(0,d,d); tt = matrix(0,d,d)
  ii = matrix(0,d,d); mm = matrix(0,d,d)
  qm = matrix(0,d,d)
  pc[1,2:d] = rbeta(d-1,a1,b1)
  r[1,2:d] = pc[1,2:d]; r[2:d,1] = pc[1,2:d]
  tt[1,2:d] = r[1,2:d]
  ss[1,2:d] = sqrt(1-r[1,2:d]^2)
  # row 2
  for(j in 3:d)
  { mm[2,j] = ss[1,2]*ss[1,j];
    ii[2,j] = tt[1,2]*tt[1,j];
    qq = max(-ii[2,j]/mm[2,j],-1)
    qm[2,j] = qq
    if(qq>=1) 
    { #cat(qq, "**limit>=1 row 2", "\n");
      r[d-1,d] = qq
      return(list(rr=r,pc=pc,qm=qm)) 
    }
    pc[2,j] = rBeta_shift(qq,mu=muvec[2],afix=afix)
    r[2,j] = ii[2,j] + pc[2,j]*mm[2,j]
    r[j,2] = r[2,j]
    ss[2,j] = ss[1,j]*sqrt(1-pc[2,j]^2)
    tt[2,j] = ss[1,j]*pc[2,j]
  }
  # row 3 ell=3
  for(j in 4:d)
  { mm[3,j] = ss[2,3]*ss[2,j];
    ii[3,j] = sum(tt[1:2,3]*tt[1:2,j])
    qq = max(-ii[3,j]/mm[3,j],-1)
    qm[3,j] = qq
    if(qq>=1) 
    { # cat(qq, "**limit>=1 row 3", "\n"); 
      r[d-1,d] = qq
      return(list(rr=r,pc=pc,qm=qm)) 
    }
    pc[3,j] = rBeta_shift(qq,mu=muvec[3],afix=afix)
    r[3,j] = ii[3,j] + pc[3,j]*mm[3,j]
    r[j,3] = r[3,j]
    ss[3,j] = ss[2,j]*sqrt(1-pc[3,j]^2)
    tt[3,j] = ss[2,j]*pc[3,j]
  }
  if(d>4)
  { for(ell in 4:(d-1))  # row ell
    { for(j in (ell+1):d)
      { mm[ell,j] = ss[ell-1,ell]*ss[ell-1,j];
        ii[ell,j] = sum(tt[1:(ell-1),ell]*tt[1:(ell-1),j])
        qq = max(-ii[ell,j]/mm[ell,j],-1)
        qm[ell,j] = qq
        if(qq>=1) 
        { message(paste(qq, "**limit>=1 row", ell)) 
          r[d-1,d] = qq
          return(list(rr=r,pc=pc,qm=qm)) 
        }
        pc[ell,j] = rBeta_shift(qq,mu=muvec[ell],afix=afix)
        r[ell,j] = ii[ell,j] + pc[ell,j]*mm[ell,j]
        r[j,ell] = r[ell,j]
        ss[ell,j] = ss[ell-1,j]*sqrt(1-pc[ell,j]^2)
        tt[ell,j] = ss[ell-1,j]*pc[ell,j]
      }
    }
  }
  if(ipermute) 
  { iperm = sample(d)
    r = r[iperm,iperm]
  }
  if(iprint)
  { cat("ii,mm,qm,pc,r,eigen\n")
    print(ii)
    print(mm)
    print(qm)
    print(pc)
    print(r)
    print(eigen(r)$values)
  }
  list(rr=r,pc=pc,qm=qm)
}

