# Functions for random correlation matrices

#' Generate correlation matrix based on partial correlation C-vine
#'
#' @description 
#' Generate correlation matrix based on partial correlation C-vine
#'
#' @param pc  dxd array with partial correlation rho_[jk;1:(j-1)] in position (j,k) for j<k
#'
#' @return dxd correlation matrix
#'
#' @examples
#' # See help pages for rcormat_posC and rcormat_mp1C
#'
#' @export
#'
pcor2cor_cvine = function(pc)
{ d = nrow(pc)
  rr = matrix(0,d,d)
  diag(rr) = 1
  rr[1,2:d] = pc[1,2:d]
  rr[2:d,1] = pc[1,2:d]
  for(j in 2:(d-1))
  { for(k in (j+1):d)
    { tem = pc[j,k]
      for(m in (j-1):1)
      { tem = pc[m,j]*pc[m,k] + tem*sqrt((1-pc[m,j]^2)*(1-pc[m,k]^2)) }
      rr[j,k] = tem; rr[k,j] = rr[j,k]
    }
  }
  rr
}

#======================================================================

# 1. Start with a d-dimensional partial correlation C-vine.
# 2. Let a_1,...,a_{d-1} be positive parameters for trees 1 to d-1.
# 3. For tree 1, generate independent correlations from a distribution with parameter a_1.
# 4. For tree ell (2<=ell<=d-1), generate independent partial correlations from a distribution with parameter a_ell.
# 5. Convert to partial correlations to a dxd correlation matrix R0
# 6. Permute the indices 1:d to get perm=(i_1,...,i_d)
# 7. Return R = R0[perm,perm] with permuted rows/columns.
#   R is always positive definite.


# Generate dxd correlation matrix with all positive correlations via positive partial correlation C-vine.
# By allowing for permutations of row/column indices, all correlations
# have the same marginal density, but one cannot derives its distribution
# in exact form. Because of the permutation step, all correlation matrices
# with all positive correlations are possible because there is always
# at least one permutation for which partial correlation C-vine has
# all non-negative entries?

#' Generate dxd correlation matrix with all positive correlations via positive partial correlation C-vine.
#'
#' @description 
#' Random positive correlation matrix via positive partial correlation C-vine
#'
#' @param d dimension >=3
#' @param avec vector of dimension d-1, first parameter of Beta(a_ell,b_ell) by tree 1 to d-1
#' @param bvec vector of dimension d-1, second parameter Beta distribution by tree 1 to d-1
#' @param ipermute permute flag, if TRUE random row/column permutation is applied
#'
#' @return dxd positive correlation matrix;
#' See rposcorr_1mom and rposcorr_2mom to generate many correlation matrices via Fortran90 
#'
#' @examples
#' # positive given mu
#' d = 4
#' mat = matrix(1:d^2,d,d)
#' iupper = upper.tri(mat)
#' a1 = 1; b1 = 3
#' mubobj = getbetapars_pos(a1,b1,d-1,iprint=FALSE)
#' print(mubobj)
#' avec = rep(a1,d-1)
#' bvec = mubobj[,'bvec']
#' nsim = 100
#' set.seed(1234)
#' out1_pos = matrix(0,nsim,d*(d-1)/2)
#' for(isim in 1:nsim)
#' { rr = rcormat_posC(d,avec,bvec,ipermute=TRUE)
#'   out1_pos[isim,] = rr[iupper]
#' }
#' colnames(out1_pos) = c("r12","r13","r23","r14","r24","r34")
#' print(rr)
#' print(summary(out1_pos))
#' print(apply(out1_pos,2,sd))
#' # positive given 2 moments
#' a1 = 4; b1 = 8
#' mubobj = get2betapars_pos(a1,b1,d-1,iprint=FALSE)
#' print(mubobj)
#' avec = mubobj[,'a']
#' bvec = mubobj[,'b']
#' nsim = 100
#' set.seed(1234)
#' out2_pos = matrix(0,nsim,d*(d-1)/2)
#' for(isim in 1:nsim)
#' { rr = rcormat_posC(d,avec,bvec,ipermute=TRUE)
#'   out2_pos[isim,] = rr[iupper]
#' }
#' colnames(out2_pos) = c("r12","r13","r23","r14","r24","r34")
#' print(summary(out2_pos))
#' print(apply(out2_pos,2,sd))
#'
#' @export
#'
rcormat_posC = function(d, avec, bvec, ipermute=TRUE)
{ # d>=3
  d = floor(d)
  if(d<3) { message("d should be >=3"); return(0) }
  # cat("in rcormat avec=", avec,"\n")
  # cat("in rcormat bvec=", bvec,"\n")
  pcmat = matrix(0,d,d)
  for(ell in 1:(d-1))
  { b = bvec[ell]
    a = avec[ell]
    r = rbeta(d-ell,a,b) # random vector for tree ell
    pcmat[ell,(ell+1):d] = r 
  }
  rmat = pcor2cor_cvine(pcmat)
  if(ipermute) 
  { iperm = sample(d)
    rmat = rmat[iperm,iperm]
  }
  return(rmat)
}


#' Generate dxd correlation matrix with partial correlation C-vine.
#'
#' @description
#' Random correlation matrix via partial correlation C-vine
#'
#' @param d dimension >=3
#' @param avec vector of dimension d-1, first parameter of Beta(a_ell,b_ell) in (-1,1) by tree 1 to d-1
#' @param bvec vector of dimension d-1, second parameter Beta distribution in (-1,1) by tree 1 to d-1
#' @param ipermute permute flag, if TRUE random row/column permutation is applied
#'
#' @return dxd positive correlation matrix, if TRUE row/column permutation is applied 
#' See rmp1corr_1mom and rmp1corr_2mom to generate many correlation matrices via Fortran90 
#'
#' @examples
#' # correlations in (-1,1) given mu
#' d = 4
#' mat = matrix(1:d^2,d,d)
#' iupper = upper.tri(mat)
#' a1 = 3; b1 = 2
#' mubobj = getbetapars_mp1(a1,b1,d-1,iprint=FALSE)
#' print(mubobj)
#' avec = rep(a1,d-1)
#' bvec = mubobj[,'bvec']  
#' nsim = 500
#' set.seed(1234)
#' out1_mp1 = matrix(0,nsim,d*(d-1)/2)
#' for(isim in 1:nsim)
#' { rr = rcormat_mp1C(d,avec,bvec,ipermute=TRUE)
#'   out1_mp1[isim,] = rr[iupper]
#' }
#' colnames(out1_mp1) = c("r12","r13","r23","r14","r24","r34")
#' print(rr)
#' print(summary(out1_mp1))
#' print(apply(out1_mp1,2,sd))
#' # mp1 given 2 moments
#' a1 = 8; b1 = 5
#' mubobj = get2betapars_mp1(a1,b1,d-1,iprint=FALSE)
#' print(mubobj)
#' avec = mubobj[,'a']
#' bvec = mubobj[,'b']
#' nsim = 500
#' set.seed(1234)
#' out2_mp1 = matrix(0,nsim,d*(d-1)/2)
#' for(isim in 1:nsim)
#' { rr = rcormat_mp1C(d,avec,bvec,ipermute=TRUE)
#'   out2_mp1[isim,] = rr[iupper]
#' }
#' colnames(out2_mp1) = c("r12","r13","r23","r14","r24","r34")
#' print(summary(out2_mp1))
#' print(apply(out2_mp1,2,sd))
#'
#' @export
#'
rcormat_mp1C = function(d, avec, bvec, ipermute=TRUE)
{ # d>=3
  d = floor(d)
  if(d<3) { message("d should be >=3"); return(0) }
  # cat("in rcormat avec=", avec,"\n")
  # cat("in rcormat bvec=", bvec,"\n")
  pcmat = matrix(0,d,d)
  for(ell in 1:(d-1))
  { b = bvec[ell]
    a = avec[ell]
    r = 2*rbeta(d-ell,a,b)-1 # random vector Beta(a,b) on (-1,1), tree ell
    pcmat[ell,(ell+1):d] = r 
  }
  rmat = pcor2cor_cvine(pcmat)
  if(ipermute) 
  { iperm = sample(d)
    rmat = rmat[iperm,iperm]
  }
  return(rmat)
}


