# Simulate with C-vine method 1: mp1 range (-1,1) or pos range (0,1)
# given 1 moment mu, or given 2 moments
# Four R functions that make use of fortran routine simmethod1
# rposcorr_1mom = function(nsim,d,a1,b,seed,ipermute=TRUE)
# rmp1corr_1mom = function(nsim,d,a1,b,seed,ipermute=TRUE)
# rposcorr_2mom = function(nsim,d,a1,b1,seed,ipermute=TRUE)
# rmp1corr_2mom = function(nsim,d,a1,b1,seed,ipermute=TRUE)


#' Generate random correlation matrices with all positive values based on
#' C-vine with positive partial correlations, fixed mean
#' 
#' @description
#' Generate random correlation matrices with all positive values based on
#' C-vine with positive partial correlations, with aim of a fixed mean
#' for all correlations  
#' 
#' @param nsim number of simulations
#' @param d dimension of correlation matrix, >=3
#' @param a1 first parameter of Beta(a1,b) random variable in (0,1) for row 1
#' @param b second parameter of Beta(a1,b) random variable in (0,1) for row 1;
#'    aim is for mean correlation value of a1/(a1+b)
#' @param seed positive integer seed for fortran90 code
#' @param ipermute permute flag, TRUE (default) for random row/column permutation 
#'
#' @return If input (d,a1,b) is feasible, output  matrix nsim by d(d-1)/2,
#' each row has a correlation matrix with lower triangle by row;
#' r12, r13, r23, r14, ..., r[d-1,d].
#' If input (d,a1,b) is infeasible,  output is 0.
#' 
#' @details 
#' Loop is done through link to Fortran90 code
#' 
#' @examples 
#' # failed case
#' rposcorr_1mom(5,4,1.0,0.3,123)
#' # good case
#' a1 = 1.5; b = 0.5; d = 6
#' out1pos = rposcorr_1mom(1000,d,a1,b,123,1) 
#' print(apply(out1pos,2,summary))
#' print(apply(out1pos,2,sd))
#' mubobj = getbetapars_pos(a1,b,d-1,iprint=FALSE)
#' print(mubobj)
#' avec = rep(a1,d-1)
#' bvec = mubobj[,'bvec']
#' recur = recursion_pos(rep(a1,d-1),mubobj[,'bvec'])
#' print(recur)
#' s = recur[d-1,7]  # theoretical SD after permutation
#' mn = recur[d-1,6] 
#' cat("theoretical mean/s", mn,s,"\n")
#'
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines. 
#' Journal of Multivariate Analysis. https://doi.org/10.1016/j.jmva.2025.105519
#'
#' @export
#'
rposcorr_1mom = function(nsim,d,a1,b,seed,ipermute=TRUE)
{ #d>=3
  d = floor(d)
  if(d<3) { message("d should be >=3"); return(0) }
  out = getbetapars_pos(a1,b,d,iprint=FALSE)
  avec = rep(a1,d)
  bvec = out[,2]
  if(length(bvec)<(d-1)) 
  { message("input parameters a1,b cannot reach given dimension d")
    return(0)
  }
  if(length(bvec)==(d-1)) bvec = c(bvec,b) # f90 code assumes length d
  #print(bvec)
  dd = d*(d-1)/2
  ipermute = as.integer(ipermute)
  ipos = 1
  rvecmat = rep(0,nsim*dd)
  out= .Fortran("simmethod1", 
     as.integer(nsim), as.integer(d), as.double(avec), as.double(bvec),
     as.integer(ipos),  as.integer(seed),  as.integer(ipermute),
     as.integer(dd), rvecmat=as.double(rvecmat))
  rvecmat = matrix(out$rvecmat,nsim,dd)
  rvecmat
}

#' Generate random correlation matrices with C-vine partial correlations, fixed mean 
#' 
#' @description
#' Generate random correlation matrices with C-vine partial correlations 
#' with aim of a fixed mean for all correlations
#' 
#' @param nsim number of simulations
#' @param d dimension of correlation matrix, >=3
#' @param a1 first parameter of Beta(a1,b) random variable in (-1,1) for row 1
#' @param b second parameter of Beta(a1,b) random variable in (-1,1) for row 1;
#'    aim is for mean correlation value of 2*a1/(a1+b)-1
#' @param seed positive integer seed for fortran90 code
#' @param ipermute permute flag, TRUE (default) for random row/column permutation 
#'
#' @return If input (d,a1,b) is feasible, output  matrix nsim by d(d-1)/2,
#' each row has a correlation matrix with lower triangle by row;
#' r12, r13, r23, r14, ..., r[d-1,d].
#' If input (d,a1,b) is infeasible,  output is 0.
#' 
#' @details 
#' Loop is done through link to Fortran90 code
#' 
#' @examples 
#' # failed case
#' rmp1corr_1mom(5,6,2.,3.0,123)
#' # good case
#' a1 = 3.0; b = 2.0; d = 4
#' out1mp1 = rmp1corr_1mom(1000,d,a1,b,123,1) 
#' print(apply(out1mp1,2,summary))
#' print(apply(out1mp1,2,sd))
#' mubobj = getbetapars_mp1(a1,b,d-1,iprint=FALSE)
#' avec = rep(a1,d-1)
#' bvec = mubobj[,'bvec']
#' recur = recursion_mp1(rep(a1,d-1),mubobj[,'bvec'])
#' print(recur)
#' s = recur[d-1,7]  # theoretical SD after permutation
#' mn = recur[d-1,6] 
#' cat("theoretical mean/s", mn,s,"\n")
#' 
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines. 
#' Journal of Multivariate Analysis. https://doi.org/10.1016/j.jmva.2025.105519
#'
#' @export
#' 
rmp1corr_1mom = function(nsim,d,a1,b,seed,ipermute=TRUE)
{ #d>=3
  d = floor(d)
  if(d<3) { message("d should be >=3"); return(0) }
  out = getbetapars_mp1(a1,b,d,iprint=FALSE)
  avec = rep(a1,d)
  bvec = out[,2]
  if(length(bvec)<(d-1)) 
  { message("input parameters a1,b cannot reach given dimension d")
    return(0)
  }
  if(length(bvec)==(d-1)) bvec = c(bvec,b) # f90 code assumes length d
  #print(bvec)
  dd = d*(d-1)/2
  ipermute = as.integer(ipermute)
  ipos = 0
  rvecmat = rep(0,nsim*dd)
  out= .Fortran("simmethod1", 
     as.integer(nsim), as.integer(d), as.double(avec), as.double(bvec),
     as.integer(ipos),  as.integer(seed),  as.integer(ipermute),
     as.integer(dd), rvecmat=as.double(rvecmat))
  rvecmat = matrix(out$rvecmat,nsim,dd)
  rvecmat
}

#' Generate random correlation matrices with all positive values based on
#' C-vine with positive partial correlations, fixed mean and SD 
#' 
#' @description
#' Generate random correlation matrices with all positive values based on
#' C-vine with positive partial correlations, with aim of a fixed mean
#' and a fixed SD for all correlations
#' 
#' @param nsim number of simulations
#' @param d dimension of correlation matrix, >=3
#' @param a1 first parameter of Beta(a1,b1) random variable in (0,1) for row 1
#' @param b1 second parameter of Beta(a1,b1) random variable in (0,1) for row 1;
#' aim is for mean correlation value of mu=a1/(a1+b1) and variance mu*(1-mu)/(a1+b1+1)
#' @param seed positive integer seed for fortran90 code
#' @param ipermute permute flag, TRUE (default) for random row/column permutation 
#'
#' @return If input (d,a1,b1) is feasible, output  matrix nsim by d(d-1)/2,
#' each row has a correlation matrix with lower triangle by row;
#' r12, r13, r23, r14, ..., r[d-1,d].
#' If input (d,a1,b1) is infeasible,  output is 0.
#' 
#' @details 
#' Loop is done through link to Fortran90 code
#' 
#' @examples 
#' # failed case
#' rposcorr_2mom(5,4,1.0,0.5,123)
#' # good case
#' a1 = 4.0; b1 = 8.0; d = 4
#' out2pos = rposcorr_2mom(1000,d,a1,b1,123,1) 
#' print(apply(out2pos,2,summary))
#' print(apply(out2pos,2,sd))
#' thmn = a1/(a1+b1)
#' thvar = thmn*(1-thmn)/(a1+b1+1)
#' cat("theoretical mean/s", thmn,sqrt(thvar),"\n")
#' colnames(out2pos) = c("r12","r13","r23","r14","r24","r34")
#' rcor = cor(out2pos)
#' print(round(rcor,3))
#' par(mfrow=c(3,3))
#' qqbeta(out2pos[,1],a1,b1,1,2)
#' qqbeta(out2pos[,2],a1,b1,1,3)
#' qqbeta(out2pos[,3],a1,b1,2,3)
#' qqbeta(out2pos[,4],a1,b1,1,4)
#' qqbeta(out2pos[,5],a1,b1,2,4)
#' qqbeta(out2pos[,6],a1,b1,3,4)
#' 
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines. 
#' Journal of Multivariate Analysis. https://doi.org/10.1016/j.jmva.2025.105519
#'
#' @export
#' 
rposcorr_2mom = function(nsim,d,a1,b1,seed,ipermute=TRUE)
{ #d>=3
  d = floor(d)
  if(d<3) { message("d should be >=3"); return(0) }
  out = get2betapars_pos(a1,b1,d)
  avec = out[,1]
  bvec = out[,2]
  # check on condition for 2 moments
  if(bvec[d-1]<=0.) 
  { message("input parameters a1,b1 cannot reach given dimension d")
    return(0)
  }
  #print(bvec)
  dd = d*(d-1)/2
  ipermute = as.integer(ipermute)
  ipos = 1
  rvecmat = rep(0,nsim*dd)
  out= .Fortran("simmethod1", 
     as.integer(nsim), as.integer(d), as.double(avec), as.double(bvec),
     as.integer(ipos),  as.integer(seed),  as.integer(ipermute),
     as.integer(dd), rvecmat=as.double(rvecmat))
  rvecmat = matrix(out$rvecmat,nsim,dd)
  rvecmat
}

#' Generate random correlation matrix with C-vine partial correlations, 
#' with fixed mean and SD 
#' 
#' @description
#' Generate random correlation matrix with C-vine partial correlations 
#' with aim of a fixed mean and a fixed SD for all correlations
#' 
#' @param nsim number of simulations
#' @param d dimension of correlation matrix, >=3
#' @param a1 first parameter of Beta(a1,b1) random variable in (-1,1) for row 1
#' @param b1 second parameter of Beta(a1,b1) random variable in (-1,1) for row 1;
#' aim is for mean correlation value of 2*mu0-1 and variance 4*mu0*(1-mu0)/(a1+b1+1)
#' where mu0 = a1/(a1+b1)
#' @param seed positive integer seed for fortran90 code
#' @param ipermute permute flag, TRUE (default) for random row/column permutation 
#'
#' @return If input (d,a1,b1) is feasible, output  matrix nsim by d(d-1)/2,
#' each row has a correlation matrix with lower triangle by row;
#' r12, r13, r23, r14, ..., r[d-1,d].
#' If input (d,a1,b1) is infeasible,  output is 0.
#' 
#' @details 
#' Loop is done through link to Fortran90 code
#' 
#' @examples 
#' # failed case
#' rmp1corr_2mom(5,4,1.0,0.6,123)
#' # good case
#' a1 = 8.0; b1 = 5.0; d = 4
#' out2mp1 = rmp1corr_2mom(1000,4,a1,b1,123,1) 
#' print(apply(out2mp1,2,summary))
#' print(apply(out2mp1,2,sd))
#' thmn = a1/(a1+b1)
#' thvar = thmn*(1-thmn)/(a1+b1+1)
#' cat("theoretical mean/s", 2*thmn-1, 2*sqrt(thvar),"\n") 
#' colnames(out2mp1) = c("r12","r13","r23","r14","r24","r34")
#' rcor = cor(out2mp1)
#' print(round(rcor,3))
#' par(mfrow=c(3,3))
#' qqbeta(out2mp1[,1],a1,b1,1,2)
#' qqbeta(out2mp1[,2],a1,b1,1,3)
#' qqbeta(out2mp1[,3],a1,b1,2,3)
#' qqbeta(out2mp1[,4],a1,b1,1,4)
#' qqbeta(out2mp1[,5],a1,b1,2,4)
#' qqbeta(out2mp1[,6],a1,b1,3,4)
#' 
#' @references 
#' Joe and Kurowicka (2026), Random correlation matrices generated via partial correlation C-vines. 
#' Journal of Multivariate Analysis. https://doi.org/10.1016/j.jmva.2025.105519
#'
#' @export
#'
rmp1corr_2mom = function(nsim,d,a1,b1,seed,ipermute=TRUE)
{ #d>=3
  d = floor(d)
  if(d<3) { message("d should be >=3"); return(0) }
  out = get2betapars_mp1(a1,b1,d)
  avec = out[,1]
  bvec = out[,2]
  if(bvec[d-1]<=0.) 
  { message("input parameters a1,b1 cannot reach given dimension d")
    return(0)
  }
  # check on condition for 2 moments
  #print(bvec)
  dd = d*(d-1)/2
  ipermute = as.integer(ipermute)
  ipos = 0
  rvecmat = rep(0,nsim*dd)
  out= .Fortran("simmethod1", 
     as.integer(nsim), as.integer(d), as.double(avec), as.double(bvec),
     as.integer(ipos),  as.integer(seed),  as.integer(ipermute),
     as.integer(dd), rvecmat=as.double(rvecmat))
  rvecmat = matrix(out$rvecmat,nsim,dd)
  rvecmat
}

